/**
 * Upgram IA Metabox JavaScript - Versão Simplificada
 * 
 * Gerencia a interação da metabox do Assistente de Conteúdo com IA
 * 
 * @package Upgram
 * @since 1.0.0
 */

// Log imediato
console.log('Upgram IA: Arquivo JavaScript carregado');

// Aguardar jQuery e DOM
(function() {
    'use strict';
    
    function init() {
        console.log('Upgram IA: Iniciando...');
        
        if (typeof jQuery === 'undefined') {
            console.error('Upgram IA: jQuery não disponível');
            return;
        }
        
        if (typeof window.upgramIAData === 'undefined') {
            console.error('Upgram IA: Dados não disponíveis');
            return;
        }
        
        const $ = jQuery;
        
        // Objeto principal
        const UpgramIA = {
            init: function() {
                console.log('Upgram IA: Inicializando...');
                this.bindEvents();
            },
            
            bindEvents: function() {
                const self = this;
                
                // Botão gerar
                $('#upgram-ia-generate-btn').on('click', function(e) {
                    e.preventDefault();
                    console.log('Upgram IA: Botão clicado');
                    self.generateContent();
                });
                
                // Botão aplicar
                $('#upgram-ia-apply-btn').on('click', function(e) {
                    e.preventDefault();
                    self.applyContent();
                });
                
                // Botão copiar
                $('#upgram-ia-copy-btn').on('click', function(e) {
                    e.preventDefault();
                    self.copyContent();
                });
                
                // Botão descartar
                $('#upgram-ia-discard-btn').on('click', function(e) {
                    e.preventDefault();
                    self.discardResult();
                });
            },
            
            generateContent: function() {
                const self = this;
                
                if (!window.upgramIAData || !window.upgramIAData.restUrl) {
                    this.showError('Erro: Dados de configuração não encontrados.');
                    return;
                }
                
                const data = {
                    post_id: parseInt(window.upgramIAData.postId),
                    user_prompt: $('#upgram-ia-user-prompt').val() || '',
                    use_current_content: $('#upgram-ia-use-current').is(':checked'),
                    tone: $('#upgram-ia-tone').val() || 'persuasive',
                    keywords: $('#upgram-ia-keywords').val() || '',
                    fields: []
                };
                
                $('input[name="upgram-ia-fields[]"]:checked').each(function() {
                    data.fields.push($(this).val());
                });
                
                if (data.fields.length === 0) {
                    this.showError('Por favor, selecione pelo menos um campo para atualizar.');
                    return;
                }
                
                $('#upgram-ia-generate-btn').prop('disabled', true);
                $('#upgram-ia-loading').show();
                $('#upgram-ia-result').hide();
                $('#upgram-ia-error').hide();
                
                $.ajax({
                    url: window.upgramIAData.restUrl,
                    method: 'POST',
                    beforeSend: function(xhr) {
                        xhr.setRequestHeader('X-WP-Nonce', window.upgramIAData.restNonce);
                    },
                    data: data,
                    dataType: 'json',
                    timeout: 120000,
                    success: function(response) {
                        console.log('Upgram IA: Resposta recebida', response);
                        self.handleSuccess(response);
                    },
                    error: function(xhr, status, error) {
                        console.error('Upgram IA: Erro', xhr.status, error);
                        self.handleError(xhr, status, error);
                    },
                    complete: function() {
                        $('#upgram-ia-generate-btn').prop('disabled', false);
                        $('#upgram-ia-loading').hide();
                    }
                });
            },
            
            handleSuccess: function(response) {
                let generatedContent = null;
                
                if (response.success && response.data && response.data.generated) {
                    generatedContent = response.data.generated;
                } else if (response.generated) {
                    generatedContent = response.generated;
                } else if (typeof response === 'string') {
                    generatedContent = response;
                }
                
                if (generatedContent) {
                    // Formatar conteúdo para padrão WordPress
                    generatedContent = this.formatWordPressContent(generatedContent);
                    
                    // Exibir com formatação HTML preservada
                    $('#upgram-ia-result-content').html(generatedContent);
                    $('#upgram-ia-result').show();
                    window.upgramIAGeneratedContent = { generated: generatedContent };
                } else {
                    this.showError('Não foi possível gerar o conteúdo. Resposta em formato inesperado.');
                }
            },
            
            formatWordPressContent: function(content) {
                // Remover markdown code blocks se existirem
                content = content.replace(/```[\s\S]*?```/g, '');
                
                // Remover explicações ou comentários que possam ter sido adicionados pela IA
                content = content.replace(/^(Explicação|Nota|Observação|Comentário|Nota:|Observação:):.*$/gmi, '');
                content = content.replace(/^<!--.*?-->/gms, ''); // Remover comentários HTML
                
                // Converter títulos markdown para HTML primeiro
                content = content.replace(/^###\s+(.+)$/gm, '<h3>$1</h3>');
                content = content.replace(/^##\s+(.+)$/gm, '<h2>$1</h2>');
                content = content.replace(/^#\s+(.+)$/gm, '<h1>$1</h1>');
                
                // Converter listas markdown para HTML
                const lines = content.split('\n');
                let inList = false;
                let listType = 'ul';
                let formattedLines = [];
                
                for (let i = 0; i < lines.length; i++) {
                    const line = lines[i].trim();
                    
                    // Verificar se é item de lista
                    const listMatch = line.match(/^[\*\-\+]\s+(.+)$/);
                    const orderedMatch = line.match(/^\d+\.\s+(.+)$/);
                    
                    if (listMatch || orderedMatch) {
                        if (!inList) {
                            listType = orderedMatch ? 'ol' : 'ul';
                            formattedLines.push('<' + listType + '>');
                            inList = true;
                        }
                        const itemText = listMatch ? listMatch[1] : orderedMatch[1];
                        formattedLines.push('<li>' + itemText + '</li>');
                    } else {
                        if (inList) {
                            formattedLines.push('</' + listType + '>');
                            inList = false;
                        }
                        if (line) {
                            formattedLines.push(line);
                        }
                    }
                }
                
                if (inList) {
                    formattedLines.push('</' + listType + '>');
                }
                
                content = formattedLines.join('\n');
                
                // Se o conteúdo não tiver tags <p>, converter texto solto em parágrafos
                if (!/<p[^>]*>/i.test(content)) {
                    // Dividir por linhas duplas ou por tags HTML
                    const blocks = content.split(/(?=<[h|u|o][^>]*>|\n\n)/);
                    content = blocks.map(block => {
                        block = block.trim();
                        if (!block) return '';
                        
                        // Se já for uma tag HTML (h1-h6, ul, ol), manter
                        if (/^<[h|u|o][^>]*>/i.test(block)) {
                            return block;
                        }
                        
                        // Se tiver múltiplas linhas, tratar cada linha
                        if (block.includes('\n')) {
                            return block.split('\n').map(line => {
                                line = line.trim();
                                if (!line) return '';
                                if (/^<[^>]+>/.test(line)) return line;
                                return '<p>' + line + '</p>';
                            }).join('\n');
                        }
                        
                        // Caso contrário, envolver em <p>
                        return '<p>' + block + '</p>';
                    }).filter(b => b).join('\n\n');
                } else {
                    // Se já tem <p>, garantir que texto solto também esteja em <p>
                    // Dividir por tags HTML e processar texto entre elas
                    const parts = content.split(/(<[^>]+>)/);
                    let result = [];
                    let currentText = '';
                    
                    for (let i = 0; i < parts.length; i++) {
                        if (/^<[^>]+>$/.test(parts[i])) {
                            // É uma tag HTML
                            if (currentText.trim()) {
                                // Adicionar texto acumulado como parágrafo
                                result.push('<p>' + currentText.trim() + '</p>');
                                currentText = '';
                            }
                            result.push(parts[i]);
                        } else {
                            // É texto
                            currentText += parts[i];
                        }
                    }
                    
                    if (currentText.trim()) {
                        result.push('<p>' + currentText.trim() + '</p>');
                    }
                    
                    content = result.join('');
                }
                
                // Limpar espaços extras e quebras de linha desnecessárias
                content = content.replace(/\n{3,}/g, '\n\n');
                content = content.replace(/>\s+</g, '><'); // Remover espaços entre tags
                content = content.replace(/>\s+/g, '>'); // Remover espaços após tags de abertura
                content = content.replace(/\s+</g, '<'); // Remover espaços antes de tags de fechamento
                content = content.trim();
                
                // Validar HTML usando jQuery
                try {
                    const $temp = $('<div>').html(content);
                    if ($temp.length > 0) {
                        content = $temp.html();
                    }
                } catch (e) {
                    console.warn('Upgram IA: Erro ao validar HTML', e);
                }
                
                return content;
            },
            
            handleError: function(xhr, status, error) {
                let errorMessage = 'Não foi possível gerar o conteúdo.';
                
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    errorMessage = xhr.responseJSON.message;
                } else if (xhr.responseText) {
                    try {
                        const response = JSON.parse(xhr.responseText);
                        errorMessage = response.message || errorMessage;
                    } catch (e) {
                        errorMessage = 'Erro HTTP ' + xhr.status;
                    }
                }
                
                this.showError(errorMessage);
            },
            
            showError: function(message) {
                $('#upgram-ia-error').text(message).show();
                $('#upgram-ia-result').hide();
            },
            
            applyContent: function() {
                if (!window.upgramIAGeneratedContent || !window.upgramIAGeneratedContent.generated) {
                    this.showError('Nenhum conteúdo gerado para aplicar.');
                    return;
                }
                
                let content = window.upgramIAGeneratedContent.generated;
                
                // Garantir que o conteúdo está formatado corretamente
                content = this.formatWordPressContent(content);
                
                // Gutenberg
                if (typeof wp !== 'undefined' && wp.data && wp.data.select('core/editor')) {
                    const dispatch = wp.data.dispatch('core/editor');
                    const select = wp.data.select('core/editor');
                    
                    if ($('input[name="upgram-ia-fields[]"][value="post_content"]').is(':checked')) {
                        // Para Gutenberg, converter HTML para blocos se necessário
                        // Por enquanto, usar HTML direto (Gutenberg aceita HTML)
                        dispatch.editPost({ content: content });
                    }
                    if ($('input[name="upgram-ia-fields[]"][value="post_title"]').is(':checked')) {
                        const title = this.extractTitle(content) || select.getEditedPostAttribute('title');
                        if (title) {
                            // Remover HTML do título
                            const titleText = $('<div>').html(title).text();
                            dispatch.editPost({ title: titleText });
                        }
                    }
                    if ($('input[name="upgram-ia-fields[]"][value="post_excerpt"]').is(':checked')) {
                        const excerpt = this.extractExcerpt(content);
                        if (excerpt) {
                            // Remover HTML do excerto
                            const excerptText = $('<div>').html(excerpt).text();
                            dispatch.editPost({ excerpt: excerptText });
                        }
                    }
                } else {
                    // Classic Editor
                    if ($('input[name="upgram-ia-fields[]"][value="post_content"]').is(':checked')) {
                        // Para editor clássico, usar HTML direto
                        $('#content').val(content);
                        if (typeof tinyMCE !== 'undefined' && tinyMCE.get('content')) {
                            // TinyMCE aceita HTML
                            tinyMCE.get('content').setContent(content);
                            // Atualizar também o textarea
                            tinyMCE.get('content').save();
                        }
                    }
                    if ($('input[name="upgram-ia-fields[]"][value="post_title"]').is(':checked')) {
                        const title = this.extractTitle(content) || $('#title').val();
                        if (title) {
                            // Remover HTML do título
                            const titleText = $('<div>').html(title).text();
                            $('#title').val(titleText).trigger('input');
                        }
                    }
                    if ($('input[name="upgram-ia-fields[]"][value="post_excerpt"]').is(':checked')) {
                        const excerpt = this.extractExcerpt(content);
                        if (excerpt) {
                            // Remover HTML do excerto
                            const excerptText = $('<div>').html(excerpt).text();
                            $('#excerpt').val(excerptText);
                        }
                    }
                }
                
                this.showSuccess('Conteúdo aplicado! Não esqueça de salvar.');
            },
            
            extractTitle: function(content) {
                const h1Match = content.match(/<h1[^>]*>(.*?)<\/h1>/i);
                if (h1Match) return h1Match[1].trim();
                const lines = content.split('\n');
                for (let i = 0; i < lines.length; i++) {
                    const line = lines[i].trim();
                    if (line && !line.match(/^<[^>]+>/)) {
                        return line.substring(0, 100);
                    }
                }
                return null;
            },
            
            extractExcerpt: function(content) {
                const text = $('<div>').html(content).text();
                const words = text.split(/\s+/);
                return words.slice(0, 150).join(' ') + (words.length > 150 ? '...' : '');
            },
            
            copyContent: function() {
                if (!window.upgramIAGeneratedContent || !window.upgramIAGeneratedContent.generated) {
                    this.showError('Nenhum conteúdo gerado para copiar.');
                    return;
                }
                const content = window.upgramIAGeneratedContent.generated;
                const textarea = $('<textarea>').val(content).appendTo('body').select();
                try {
                    document.execCommand('copy');
                    textarea.remove();
                    this.showSuccess('Conteúdo copiado!');
                } catch (err) {
                    textarea.remove();
                    this.showError('Não foi possível copiar.');
                }
            },
            
            discardResult: function() {
                $('#upgram-ia-result').hide();
                $('#upgram-ia-error').hide();
                window.upgramIAGeneratedContent = null;
            },
            
            showSuccess: function(message) {
                const $notice = $('<div>')
                    .addClass('notice notice-success is-dismissible')
                    .css({
                        'position': 'fixed',
                        'top': '32px',
                        'right': '20px',
                        'z-index': '999999',
                        'max-width': '400px'
                    })
                    .html('<p>' + message + '</p>');
                $('body').append($notice);
                setTimeout(function() {
                    $notice.fadeOut(function() {
                        $(this).remove();
                    });
                }, 3000);
            }
        };
        
        // Tornar global
        window.UpgramIA = UpgramIA;
        console.log('Upgram IA: Objeto criado');
        
        // Inicializar quando DOM estiver pronto
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', function() {
                UpgramIA.init();
            });
        } else {
            // DOM já pronto, mas aguardar um pouco para garantir que os elementos existam
            setTimeout(function() {
                UpgramIA.init();
            }, 100);
        }
    }
    
    // Tentar iniciar
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        setTimeout(init, 100);
    }
})();

